<?php

namespace App\Livewire\User;

use App\Models\UserAiSettings;
use App\Services\AiProviderFactory;
use Livewire\Attributes\On;
use Livewire\Component;

class AiSettingsModal extends Component
{
    public bool $showModal = false;

    public string $preferred_provider = 'system';

    public ?string $preferred_model = null;

    public ?string $openai_api_key = null;

    public ?string $anthropic_api_key = null;

    public bool $canUseOwnKey = false;

    public array $availableProviders = [];

    public array $availableModels = [];

    public function mount()
    {
        $this->loadSettings();
    }

    #[On('open-ai-settings-modal')]
    public function open()
    {
        $this->loadSettings();
        $this->showModal = true;
    }

    private function loadSettings()
    {
        $user = auth()->user();
        $this->canUseOwnKey = $user->canUseOwnAiApiKey();

        $settings = $user->aiSettings;
        if ($settings) {
            $this->preferred_provider = $settings->preferred_provider;
            $this->preferred_model = $settings->preferred_model;
            $this->openai_api_key = $settings->openai_api_key ? '••••••••••••••••' : null;
            $this->anthropic_api_key = $settings->anthropic_api_key ? '••••••••••••••••' : null;
        } else {
            $this->preferred_provider = 'system';
            $this->preferred_model = null;
            $this->openai_api_key = null;
            $this->anthropic_api_key = null;
        }

        $this->availableProviders = AiProviderFactory::getAvailableProviders();
        $this->loadModelsForProvider();
    }

    public function updatedPreferredProvider()
    {
        $this->loadModelsForProvider();
        $this->preferred_model = null;
    }

    private function loadModelsForProvider()
    {
        if ($this->preferred_provider === 'system') {
            $this->availableModels = [];

            return;
        }

        $this->availableModels = AiProviderFactory::getModelsForProvider($this->preferred_provider);
    }

    public function save()
    {
        if (! $this->canUseOwnKey) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('common.no_own_api_key')
            );

            return;
        }

        $user = auth()->user();
        $settings = $user->aiSettings ?? new UserAiSettings(['user_id' => $user->id]);

        $settings->preferred_provider = $this->preferred_provider;
        $settings->preferred_model = $this->preferred_provider !== 'system' ? $this->preferred_model : null;

        if ($this->preferred_provider === 'openai') {
            if ($this->openai_api_key && ! str_starts_with($this->openai_api_key, '•')) {
                $settings->openai_api_key = $this->openai_api_key;
            } elseif (! $this->openai_api_key) {
                $settings->openai_api_key = null;
            }
        }

        if ($this->preferred_provider === 'anthropic') {
            if ($this->anthropic_api_key && ! str_starts_with($this->anthropic_api_key, '•')) {
                $settings->anthropic_api_key = $this->anthropic_api_key;
            } elseif (! $this->anthropic_api_key) {
                $settings->anthropic_api_key = null;
            }
        }

        $settings->save();

        if ($settings->openai_api_key) {
            $this->openai_api_key = '••••••••••••••••';
        }
        if ($settings->anthropic_api_key) {
            $this->anthropic_api_key = '••••••••••••••••';
        }

        $this->dispatch('show-toast',
            type: 'success',
            message: __('common.item_saved', ['item' => 'AI settings'])
        );

        $this->closeModal();
    }

    public function testOpenAiKey()
    {
        if (! $this->openai_api_key || str_starts_with($this->openai_api_key, '•')) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('common.enter_new_api_key')
            );

            return;
        }

        try {
            $provider = AiProviderFactory::createProvider('openai', $this->openai_api_key);
            $success = $provider->testConnection($this->openai_api_key);

            if ($success) {
                $this->dispatch('show-toast',
                    type: 'success',
                    message: __('common.api_key_valid', ['provider' => 'OpenAI'])
                );
            } else {
                $this->dispatch('show-toast',
                    type: 'error',
                    message: __('common.api_key_invalid', ['provider' => 'OpenAI'])
                );
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('common.connection_failed', ['error' => $e->getMessage()])
            );
        }
    }

    public function testAnthropicKey()
    {
        if (! $this->anthropic_api_key || str_starts_with($this->anthropic_api_key, '•')) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('common.enter_new_api_key')
            );

            return;
        }

        try {
            $provider = AiProviderFactory::createProvider('anthropic', $this->anthropic_api_key);
            $success = $provider->testConnection($this->anthropic_api_key);

            if ($success) {
                $this->dispatch('show-toast',
                    type: 'success',
                    message: __('common.api_key_valid', ['provider' => 'Anthropic'])
                );
            } else {
                $this->dispatch('show-toast',
                    type: 'error',
                    message: __('common.api_key_invalid', ['provider' => 'Anthropic'])
                );
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast',
                type: 'error',
                message: __('common.connection_failed', ['error' => $e->getMessage()])
            );
        }
    }

    public function clearOpenAiKey()
    {
        $this->openai_api_key = null;
    }

    public function clearAnthropicKey()
    {
        $this->anthropic_api_key = null;
    }

    public function closeModal()
    {
        $this->showModal = false;
    }

    public function render()
    {
        return view('livewire.user.ai-settings-modal');
    }
}
